<?php

namespace App\Restify\Users;

use App\Enums\Permissions;
use App\Models\User;
use App\Restify\Permissions\RoleRepository;
use App\Restify\Repository;
use App\Restify\TeamRepository;
use App\Restify\Users\Actions\AttachRoleFieldAction;
use Binaryk\LaravelRestify\Fields\BelongsToMany;
use Binaryk\LaravelRestify\Http\Requests\RestifyRequest;
use Illuminate\Validation\Rule;

/**
 * @property-read User $resource
 */
class UserRepository extends Repository
{
    public static string $model = User::class;

    public static array $search = [
        'name',
        'email',
    ];

    public static array $match = [
        'id' => 'int',
        'email' => 'string',
    ];

    public static function related(): array
    {
        return [
            BelongsToMany::make('teams', TeamRepository::class),
            BelongsToMany::make('roles', RoleRepository::class),
        ];
    }

    public function fields(RestifyRequest $request): array
    {
        return [
            field('phone'),

            field('name')->storingRules('required'),

            field('email')
                ->storingRules('required', 'unique:users')
                ->updatingRules(Rule::unique('users')->ignore($this->resource->id)),

            field('password')
                ->storingRules('required', 'confirmed')
                ->updatingRules('nullable', 'confirmed')
                ->messages([
                    'required' => 'This field is required.',
                    'confirmed' => 'The password confirmation does not match.',
                ])
                ->hideFromIndex()
                ->hideFromShow(),

            field('role', fn () => $this->resource->roles->pluck('name')->implode(', '))
                ->canSee(fn () => $request->user()?->can(Permissions::manageUsers->value) || $this->resource->is($request->user()))
                ->action(app(AttachRoleFieldAction::class)),
        ];
    }
}
