<?php

namespace App\Models;

use App\Mail\InvitationMail;
use App\Models\Concerns\WithCreator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;

/**
 * Class Invitation
 *
 * @property string $id
 * @property string $email,
 * @property string|null $token
 * @property Carbon|null $validated_at
 * @property Carbon|null $cancelled_at
 * @property-read User $creator,
 * @property-read User $user
 * @property-read Role $role
 */
class Invitation extends Model
{
    use WithCreator;

    protected $table = 'invitations';

    protected $fillable = [
        'email',
        'token',
        'user_id',
        'role_id',
        'validated_at',
        'validated_at',
    ];

    protected $casts = [
        'validated_at' => 'datetime',
        'cancelled_at' => 'datetime',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function role(): BelongsTo
    {
        return $this->belongsTo(Role::class, 'role_id');
    }

    public function validate(): self
    {
        $this->token = null;
        $this->validated_at = now();
        $this->save();

        return $this;
    }

    public function cancel(): self
    {
        $this->token = null;
        $this->cancelled_at = now();
        $this->save();

        return $this;
    }

    public function reactivate(): self
    {
        $this->populateToken();
        $this->cancelled_at = null;
        $this->save();

        return $this;
    }

    public function sendInvitationEmail(): void
    {
        if ($this->isCancelled() || $this->isValidated()) {
            return;
        }

        if (empty($this->token)) {
            $this->populateToken()->save();
        }

        Mail::to($this->email)->send(new InvitationMail($this));
    }

    public function isCancelled(): bool
    {
        return (bool) $this->cancelled_at;
    }

    public function isValidated(): bool
    {
        return (bool) $this->validated_at;
    }

    public function populateToken(): self
    {
        $this->token = Str::random(64);

        return $this;
    }

    public function invitationUrl(): string
    {
        return config('app.url').'/invitation/'.$this->token.'?email='.urlencode($this->email);
    }
}
